#include <stdio.h>
#include <stdbool.h>
#include <stdint.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>

typedef struct square{
	bool hypo[9]; // Hypothesis tabular
	uint8_t value; //value
	uint8_t n_hypo;
}square;

static square game[9*9]; // Game is composed from 81 squares
static uint16_t square_to_solve;

// Prototypes
int main();
void clear_game();
void load_game();
void read_database();
void print_game();
void print_hyp();
void solve_game();
uint8_t check_hypotheses(square);
bool  line_tests();
bool column_tests();
bool  square_tests();

// Getters and setters
uint8_t get_value(uint16_t,uint16_t);
void set_value(uint16_t,uint16_t,uint8_t);
bool get_hypo(uint16_t,uint16_t,uint16_t);
void hypo_false(uint16_t,uint16_t,uint16_t);
void one_hypo(uint16_t,uint16_t);


int main(){
	char c;
  
	struct timeval tv;
	printf("=== THIS IS NITRA SUDOKU SOLVER BY JUSTIN CANO === \n");
	printf("Version 0.2, july 2018 \n");
	GAME:
	clear_game();
	
	load_game();
	
	printf("Initial grid : \n");
	print_game();
	
	// Solve the game and see how long it takes
	gettimeofday(&tv,NULL);
	uint64_t micros = tv.tv_usec;
    solve_game();
	gettimeofday(&tv,NULL);
	uint64_t micros2= tv.tv_usec;
	
	printf("-------- \n");
	printf("The solution proposed is ");
	if(square_to_solve == 0){
		printf("complete, all the squares are solved :) \n");
	}
	else{
		printf("incomplete, %d squares are missing because there are \n too few constraints :( \n",square_to_solve);
		printf("I think that the solution is not unique... \nThus you can argue with your newspaper vendor or blame yourself for a typo ! \n");
		printf("Here are the remaining possibilities for each squares : \n");
		print_hyp();
		printf("---- \n");
	}
	
	
    printf("Time to solve the problem = %lu microseconds \n",micros2-micros);
    printf("Here is the solved grid \n");
	print_game();
	
	printf("Another game? Press 'y' to continue or 'n' to exit this program \n");
	scanf(" %c",&c);
	if(c=='y')
		goto GAME;
	return 0;
}

void clear_game(){
	uint16_t i,j;
	for(i=0;i<81;i++){
		game[i].value = 0; // A priori hypothesis blank (undetermined value)
		game[i].n_hypo = 9; // 9 hypothesis per square
		for(j=0;j<9;j++){
			game[i].hypo[j]= true; // All the hypotheses are valid
		}
	}
	square_to_solve = 81;
}

void load_game(){
	char e = 0;
	
	while(1){
	printf("Enter 'd' for a demo or 'r' to read a sudoku database \n");
	scanf(" %c",&e);
	switch(e){
	case 'd':
		printf("DEMO SUDOKU \n");
		set_value(1,1,5);
		set_value(1,2,3);
		set_value(1,5,7);
	
		set_value(2,1,6);
		set_value(2,4,1);
		set_value(2,5,9);
		set_value(2,6,5);
	
		set_value(3,2,9);
		set_value(3,3,8);
		set_value(3,8,6);
	
		set_value(4,1,8);
		set_value(4,5,6);
		set_value(4,9,3);
	
		set_value(5,1,4);
		set_value(5,4,8);
		set_value(5,6,3);
		set_value(5,9,1);
	
		set_value(6,1,7);
		set_value(6,5,2);
		set_value(6,9,6);
	
		set_value(7,2,6);
		set_value(7,7,2);
		set_value(7,8,8);
	
		set_value(8,4,4);
		set_value(8,5,1);
		set_value(8,6,9);
		set_value(8,9,5);
	
		set_value(9,5,8);
		set_value(9,8,7);
		set_value(9,9,9);
		break;
	case 'r':
		printf("Handling database...");
		read_database();
		break;
	}
	break;
	}
}

void read_database(){
		uint16_t i,j,column,row;
		bool is_digit;
		uint8_t digit;
		FILE *fp;
		char * line = NULL;
		size_t len = 0;
		ssize_t read;
		
		printf("Opening sudodata.txt ...");
		// Read
		fp = fopen("sudodata.txt","r");
		if(fp == NULL){
			printf("Failure the file do not exist... try again!");
			exit(EXIT_FAILURE);
		}
		else{
			// Handling the file
			printf("\n");
			i=0;
			row =0;
			while ((read = getline(&line, &len, fp)) != -1 && i<11) {
				// For the first 11 lines (containing sudoku)
				//printf("Retrieved line of length %zu :\n", read);
				//printf("%s \n", line);
				//printf("len = %zu \n", read);
				column=0;
				for(j=0;j<read;j++){
					// For all the values of the line)
					is_digit = line[j]>='0' && line[j]<='9';
					if(is_digit || line[j]=='.'){
						if(column==0){
							row++;
						}
						// if the char is representing a digit between 1 and 9 or a dot (uncomplete value)
						column++; // we are at the k-th line of the sudoku
						if(is_digit){
							// We have to convert the digit
							//[DEBUG]
							//set_value(row,column,digit);
							//printf("row %lu, column %lu \n",row,column);
							digit = line[j] - 48; // ascii to digit conv
							set_value(row,column,digit);
						}
					}
				}
				i++;
			}
			fclose(fp);
			if (line)
				free(line);
			printf("success reading, the dataset is : \n");
			print_game();
			//exit(EXIT_SUCCESS);
		}
}


void print_game(){
	printf("--------------------------------------\n");
	uint16_t i,j;
	char buff[120];
	char s[9];
	for(i=0;i<9;i++){
		strcpy(buff,""); // clearing the buffer before concatenation
		if(i%3==0){
		 strcat(buff,"\n");
		}
		for(j=0; j<9; j++){
			if(j%3==0){
			  strcat(buff," ");
			}
			sprintf(s,"%d ",game[9*i+j].value);
			strcat(buff,s);
		}
		strcat(buff,"\n");
		printf(buff);
	}
}

void print_hyp(){
	printf("\nHypothesis map: each digit is representing the number of potential solutions :\n ------- ");
	uint16_t i,j;
	char buff[120];
	char s[9];
	for(i=0;i<9;i++){
		strcpy(buff,""); // clearing the buffer before concatenation
		if(i%3==0){
		 strcat(buff,"\n");
		}
		for(j=0; j<9; j++){
			if(j%3==0){
			  strcat(buff," ");
			}
			sprintf(s,"%d ",game[9*i+j].n_hypo);
			strcat(buff,s);
		}
		strcat(buff,"\n");
		printf(buff);
	}
}

bool line_tests(){
	bool evolution = false;
	uint8_t val,val2;
	uint16_t i, j, k;
	for(i=1; i<10;i++){
		// For each line
		for(j=1; j<10;j++){
			// For each square of this line
			val = get_value(i,j);
			if(val>0){
				
				// If the square is solved
				for(k=1; k<10;k++){
					// For all the othe non-solved
					val2 = get_value(i,k);
					if(val2==0){
						if(get_hypo(i,k,val)){
							// If the hypothesis is true, set it to false and test
							hypo_false(i,k,val);
							evolution = true;
						}
					}
				}
			}
		}
	}
	return evolution;
}

bool column_tests(){
	bool evolution = false;
	uint8_t val,val2;
	uint16_t i, j, k;
	for(i=1; i<10;i++){
		// For each column
		for(j=1; j<10;j++){
			// For each square of this column
			val = get_value(j,i);
			if(val>0){
				// If the square is solved
				for(k=1; k<10;k++){
					// For all the othe non-solved
					val2 = get_value(k,i);
					if(val2==0){
						if(get_hypo(k,i,val)){
							// If the hypothesis is true, set it to false and test
							hypo_false(k,i,val);
							evolution = true;
						}
					}
				}
			}
		}
	}
	return evolution;
}

bool square_tests(){
	bool evolution = false;
	uint16_t i, j, ii, jj, square_i, square_j, ki, kj;
	uint8_t val, val2;
	for(i=0; i<3; i++){
		for(j=0; j<3; j++){
			// First column and first line of each 3*3 square
			square_i = 1 + 3*i;
			square_j = 1 + 3*j;
			for(ii=0;ii<3;ii++){
				for(jj=0;jj<3;jj++){
					val = get_value(square_i+ii,square_j+jj);
					if(val>0){
						for(ki=0;ki<3;ki++){
							for(kj=0;kj<3;kj++){
								val2 = get_value(square_i+ki,square_j+kj);
								if(val2==0){
									if(get_hypo(square_i+ki,square_j+kj,val)){
									// If the hypothesis is true, set it to false and test
										hypo_false(square_i+ki,square_j+kj,val);
										evolution = true;
									}
								}
							}
						}
					}
				}
			}
		}
	}
	return evolution;
}

void solve_game(){
	bool a,b,c;
	int osef;
	uint16_t r;
	a = true;
	b = true;
	c = true;
	while(r<2){
		a = line_tests();
		if(square_to_solve == 0)
			break;
		//print_hyp();
		//print_game();


		b = column_tests();
		if(square_to_solve == 0)
			break;
		//print_game();

		
		c = square_tests();
		if(square_to_solve == 0)
			break;
		//print_game();
		if((a || b || c)==0){
			r++;
		} 

	}
}



uint8_t get_value(uint16_t row, uint16_t column){
	return game[(row-1)*9 +column-1].value;
}

void set_value(uint16_t row, uint16_t column, uint8_t value){
	game[(row-1)*9 +column-1].value = value;
	game[(row-1)*9 +column-1].n_hypo = 1;
	square_to_solve--;
}

bool get_hypo(uint16_t row, uint16_t column, uint16_t number){
	return game[(row-1)*9 +column-1].hypo[number-1];
}

void hypo_false(uint16_t row, uint16_t column, uint16_t number){
	game[(row-1)*9 +column-1].n_hypo--;
	game[(row-1)*9 +column-1].hypo[number-1]=false;
	if(game[(row-1)*9 +column-1].n_hypo==0){
		printf("Error, the sudoku have too much constraints! The grid seems to be false... \n");
		exit(EXIT_FAILURE);
	}
	else if(game[(row-1)*9 +column-1].n_hypo == 1){
		one_hypo(row,column);
	}
}

void one_hypo(uint16_t row, uint16_t column){
	uint8_t i;
	for(i=1;i<10;i++){
		if(get_hypo(row,column,i)){
			game[(row-1)*9 +column-1].value = i;
			square_to_solve--;
			break;
		}
	}
}
